<?php
/**
 * Manages the general setting page of Keyzy.
 */
class KeyzySettingsPage
{
    /**
     * Holds the values to be used in the fields callbacks
     */
    private $options;

    /**
     * Start up
     */
    public function __construct()
    {
        add_action('admin_menu', array($this, 'add_plugin_page'));
        add_action('admin_init', array($this, 'page_init'));
        // Enqueue CSS and JS for tooltips
        add_action('admin_enqueue_scripts', array($this, 'enqueue_tooltip_styles'));
    }

    /**
     * Enqueue CSS and JS for tooltips
     */
    public function enqueue_tooltip_styles()
    {
        // Only enqueue on the Keyzy settings page
        $screen = get_current_screen();
        if ($screen->id === 'settings_page_keyzy-setting-admin') {
            wp_enqueue_style('keyzy-tooltip-styles', plugin_dir_url(__FILE__) . 'css/keyzy-tooltip.css');
        }
    }

    /**
     * Add options page
     */
    public function add_plugin_page()
    {
        // This page will be under "Settings"
        add_options_page(
            'Keyzy Settings Admin', // Page Title
            'Keyzy Settings',       // Menu Title
            'manage_options',       // Capability
            'keyzy-setting-admin',  // Menu Slug
            array($this, 'create_admin_page') // Function
        );
    }

    /**
     * Options page callback
     */
    public function create_admin_page()
    {
        // Set class property
        $this->options = get_option('keyzy_option_name', []);
        ?>
        <div class="wrap">
            <h2>Keyzy Integrator</h2>
            <form method="post" action="options.php">
                <?php
                // This prints out all hidden setting fields
                settings_fields('keyzy_option_group');
                do_settings_sections('keyzy-setting-admin');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Register and add settings
     */
    public function page_init()
    {
        register_setting(
            'keyzy_option_group', // Option group
            'keyzy_option_name', // Option name
            array($this, 'sanitize') // Sanitize
        );

        add_settings_section(
            'keyzy_setting_section_id', // ID
            'Keyzy Custom Settings', // Title
            array($this, 'print_section_info'), // Callback
            'keyzy-setting-admin' // Page
        );

        add_settings_field(
            'app_id', // ID
            'APP ID <span class="keyzy-tooltip">i<span class="keyzy-tooltiptext">Your App ID from app.keyzy.io under App Keys section. Use the one with write (admin) permission.</span></span>', // Title with tooltip
            array($this, 'app_id_callback'), // Callback
            'keyzy-setting-admin', // Page
            'keyzy_setting_section_id' // Section
        );

        add_settings_field(
            'api_key',
            'API KEY <span class="keyzy-tooltip">i<span class="keyzy-tooltiptext">Your API Key from app.keyzy.io under App Keys section. Ensure it has write (admin) permission.</span></span>',
            array($this, 'api_key_callback'),
            'keyzy-setting-admin',
            'keyzy_setting_section_id'
        );

        add_settings_field(
            'show_download_link',
            'Show Download Link <span class="keyzy-tooltip">i<span class="keyzy-tooltiptext">Enable to display the download link for licensed products on the front-end.</span></span>',
            array($this, 'show_download_link_callback'),
            'keyzy-setting-admin',
            'keyzy_setting_section_id'
        );

        add_settings_field(
            'show_host_id_input',
            'Show Host ID Input Field <span class="keyzy-tooltip">i<span class="keyzy-tooltiptext">Enable to show a field for users to input their Host ID during download.</span></span>',
            array($this, 'show_host_id_input_callback'),
            'keyzy-setting-admin',
            'keyzy_setting_section_id'
        );

        add_settings_field(
            'show_manageable_activations_to_users',
            'Show Manageable Activations to Users <span class="keyzy-tooltip">i<span class="keyzy-tooltiptext">Enable to display the number of manageable activations to users.</span></span>',
            array($this, 'show_manageable_activations_to_users_callback'),
            'keyzy-setting-admin',
            'keyzy_setting_section_id'
        );

        add_settings_field(
            'disable_register_product_notification',
            'Disable Register Product Notification <span class="keyzy-tooltip">i<span class="keyzy-tooltiptext">Enable to turn off product registration notifications for users.</span></span>',
            array($this, 'disable_register_product_notification_callback'),
            'keyzy-setting-admin',
            'keyzy_setting_section_id'
        );
    }

    /**
     * Sanitize each setting field as needed
     *
     * @param array $input Contains all settings fields as array keys
     */
    public function sanitize($input)
    {
        $new_input = array();
        if (isset($input['app_id']))
            $new_input['app_id'] = sanitize_text_field($input['app_id']);

        if (isset($input['api_key']))
            $new_input['api_key'] = sanitize_text_field($input['api_key']);

        if (isset($input['show_download_link']))
            if (sanitize_text_field($input['show_download_link']) === 'on')
                $new_input['show_download_link'] = 1;
            else
                $new_input['show_download_link'] = 0;

        if (isset($input['show_host_id_input']))
            if (sanitize_text_field($input['show_host_id_input']) === 'on')
                $new_input['show_host_id_input'] = 1;
            else
                $new_input['show_host_id_input'] = 0;

        if (isset($input['show_manageable_activations_to_users']))
            if (sanitize_text_field($input['show_manageable_activations_to_users']) === 'on')
                $new_input['show_manageable_activations_to_users'] = 1;
            else
                $new_input['show_manageable_activations_to_users'] = 0;

        if (isset($input['disable_register_product_notification']))
            if (sanitize_text_field($input['disable_register_product_notification']) === 'on')
                $new_input['disable_register_product_notification'] = 1;
            else
                $new_input['disable_register_product_notification'] = 0;

        return $new_input;
    }

    /**
     * Print the Section text
     */
    public function print_section_info()
    {
        print 'Enter your settings below: (You can get your APPKEYS on app.keyzy.io under App Keys section. Please use write (admin) permission one.';
    }

    /**
     * Get the settings option array and print one of its values
     */
    public function app_id_callback()
    {
        printf(
            '<input type="text" id="app_id" name="keyzy_option_name[app_id]" value="%s" size="40" />',
            isset($this->options['app_id']) ? esc_attr($this->options['app_id']) : ''
        );
    }

    /**
     * Get the settings option array and print one of its values
     */
    public function api_key_callback()
    {
        printf(
            '<input type="text" id="api_key" name="keyzy_option_name[api_key]" value="%s" size="40" />',
            isset($this->options['api_key']) ? esc_attr($this->options['api_key']) : ''
        );
    }

    /**
     * Get the settings option array and print one of its values
     */
    public function show_download_link_callback()
    {
        printf(
            '<input type="checkbox" id="show_download_link" name="keyzy_option_name[show_download_link]" %s />',
            !empty($this->options['show_download_link']) ? 'checked' : ''
        );
    }

    /**
     * Get the settings option array and print one of its values
     */
    public function show_host_id_input_callback()
    {
        printf(
            '<input type="checkbox" id="show_host_id_input" name="keyzy_option_name[show_host_id_input]" %s />',
            !empty($this->options['show_host_id_input']) ? 'checked' : ''
        );
    }

    /**
     * Get the settings option array and print one of its values
     */
    public function show_manageable_activations_to_users_callback()
    {
        printf(
            '<input type="checkbox" id="show_manageable_activations_to_users" name="keyzy_option_name[show_manageable_activations_to_users]" %s />',
            !empty($this->options['show_manageable_activations_to_users']) ? 'checked' : ''
        );
    }

    /**
     * Get the settings option array and print one of its values
     */
    public function disable_register_product_notification_callback()
    {
        printf(
            '<input type="checkbox" id="disable_register_product_notification" name="keyzy_option_name[disable_register_product_notification]" %s />',
            !empty($this->options['disable_register_product_notification']) ? 'checked' : ''
        );
    }
}

if (is_admin())
    $keyzy_settings_page = new KeyzySettingsPage();
?>
