<?php
/*
 * Plugin Name: Keyzy WooCommerce Integration
 * Plugin URI: www.keyzy.io
 * Description: Integrates a WooCommerce store with Keyzy (app.keyzy.io) service.
 * Author: Volkan Ozyilmaz
 * Version: 1.4.2
 * Author URI: www.keyzy.io
 */

// Load Composer autoloader
require_once dirname(__FILE__) . '/vendor/autoload.php';

// Load definitions if the file exists (Best practice check)
if (file_exists(dirname(__FILE__) . '/defines.php')) {
    require_once dirname(__FILE__) . '/defines.php';
}

// Declare HPOS (High-Performance Order Storage) compatibility for WooCommerce 7.1+
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
            'custom_order_tables',
            __FILE__
        );
    }
});

// Load Settings Page class
require_once dirname(__FILE__) . '/KeyzySettingsPage.php';

if (!class_exists('KeyzyDownloads')) :

    class KeyzyDownloads {

        public static function init() {
            add_shortcode('keyzy-downloads', __CLASS__ . '::keyzy_downloads_shortcode');
            add_shortcode('keyzy-register-product', __CLASS__ . '::keyzy_register_product_shortcode');
            add_shortcode('keyzy-upgrade', __CLASS__ . '::keyzy_upgrade_shortcode');
        }

        /**
         * Shortcode to display Keyzy downloads
         * Usage: [keyzy-downloads]
         */
        public static function keyzy_downloads_shortcode () {
            if (is_user_logged_in()) {
                // Global WooCommerce object is usually not strictly needed here but kept for legacy compatibility
                global $woocommerce;

                ob_start();
                // Load the template file (which handles data fetching internally)
                require dirname(__FILE__) . '/tpl/keyzy-downloads.php';
                return ob_get_clean();
            } else {
                return "<h3><b>Whoops!</b></h3><p>It looks like you aren't logged in. Please Log in to see this page.</p>";
            }
        }

        /**
         * Shortcode to display Keyzy register product form
         * Usage: [keyzy-register-product]
         */
        public static function keyzy_register_product_shortcode () {
            if (is_user_logged_in()) {
                ob_start();
                require dirname(__FILE__) . '/tpl/KeyzyRegisterProduct.php';
                return ob_get_clean();
            } else {
                return "<h3><b>Whoops!</b></h3><p>It looks like you aren't logged in. Please Log in to see this page.</p>";
            }
        }

        /**
         * Shortcode to display Keyzy upgrade form
         * Usage: [keyzy-upgrade]
         */
        public static function keyzy_upgrade_shortcode () {
            if (is_user_logged_in()) {
                ob_start();
                require dirname(__FILE__) . '/tpl/KeyzyUpgrade.php';
                return ob_get_clean();
            } else {
                return "<h3><b>Whoops!</b></h3><p>It looks like you aren't logged in. Please Log in to see this page.</p>";
            }
        }

    }

    KeyzyDownloads::init();

endif;

/**
 * Hook: When a new order is completed, register the product license in Keyzy.
 */
function keyzy_take_product_serials($order_id) {
    global $connection_array; // Defined in defines.php

    $order = new WC_Order( $order_id );

    // Check if this is an internal order created by Keyzy itself to avoid infinite loops
    if ($order->get_meta('_keyzy_internal_order', true, 'view') === 'true')
        return;

    $user_id = $order->get_user_id();
    $user = get_user_by( 'id', $user_id );

    // Check if user exists
    if ( ! $user ) {
        error_log("KEYZYWc: User ID {$user_id} not found for Order #{$order_id}. Skipping license registration.");
        return;
    }

    if ( count( $order->get_items() ) > 0 )
    {
        // Get customer's app_id and api_key settings
        // NOTE: Default value is set to [] (empty array) to avoid type errors in PHP 8+
        $optionInfo = get_option( 'keyzy_option_name', [] );

        // Safe array key access
        $app_id  = isset($optionInfo['app_id']) ? $optionInfo['app_id'] : '';
        $api_key = isset($optionInfo['api_key']) ? $optionInfo['api_key'] : '';

        // If API Keys are missing, stop execution
        if (empty($app_id) || empty($api_key)) {
            error_log("KEYZYWc: API Keys missing in settings. Cannot register license.");
            return;
        }

        foreach( $order->get_items() as $item ) {
            $product_id = $item->get_product_id();
            $product = wc_get_product( $product_id );

            // Skip if product no longer exists
            if ( ! $product ) {
                error_log("KEYZYWc: Product ID {$product_id} not found. Skipping license registration.");
                continue;
            }

            $itemData = $item->get_data();
            $sku = $product->get_sku();

            // Name Resolution Logic
            // Priority: User Name -> Billing Name -> Shipping Name -> Order Number
            $name = trim($user->first_name . " " . $user->last_name);
            if (!empty($name)) {
                // Name found from user profile
            } else {
                $first_name = $order->get_billing_first_name();
                $last_name = $order->get_billing_last_name();
                $name = trim($first_name . " " . $last_name);
                if (empty($name)) {
                    $first_name = $order->get_shipping_first_name();
                    $last_name = $order->get_shipping_last_name();
                    $name = trim($first_name . " " . $last_name);
                    if (empty($name)) {
                        $name = "Order #" . $order->get_order_number();
                    }
                }
            }

            $email = $user->user_email;
            $res = null;

            try {
                // Instantiate Guzzle Client
                $client = new GuzzleHttp\Client( $connection_array );

                $itemQuantity = $itemData['quantity'];
                $pluginName = defined('KEYZYWC_VERSION') ? 'keyzywc/' . KEYZYWC_VERSION : 'keyzywc/1.0.0';
                
                // Register a license for each quantity of the item
                for ($qua = 0; $qua < $itemQuantity; ++$qua) {
                    $res = $client->request('POST', 'licenses/register', [
                        'headers' => [
                            'plugin-name' => $pluginName
                        ],
                        'form_params' => [
                            'app_id' => $app_id,
                            'api_key' => $api_key,
                            'meta' => [ 
                                'user_id' => $user_id,
                                'order_id' => $order_id,
                                'sku_number' => $sku
                            ],
                            'sku_number' => $sku,
                            'name' => $name,
                            'email' => $email
                        ]
                    ]);
                    
                    // Log response status
                    if ($res) {
                        $retArray = json_decode($res->getBody(), true);
                        if ( isset( $retArray['error'] ) ) {
                            error_log('KEYZYWc Error: ' . print_r($retArray['error']['message'], true));
                        } else {
                            error_log("KEYZYWc: License registered for Order #".$order_id." SKU: ".$sku);
                        }
                    }
                }
            } catch (Exception $e) {
                error_log('KEYZYWc Exception: ' . $e->getMessage());
            }
        }
    }
}
add_action( 'woocommerce_order_status_completed', 'keyzy_take_product_serials' );


/**
 * Filter: Suppress notification emails generated by the internal 'Register Product' function.
 */
function filter_woocommerce_email_recipient_keyzy_new_order( $recipient, $order ) {
    if ( ! $order || ! is_a( $order, 'WC_Order' ) ) return $recipient;

    if ( $order->get_meta('_keyzy_internal_order', true, 'view') === 'true' ) {
        // Retrieve settings safely
        $optionInfo = get_option( 'keyzy_option_name', [] );

        // Check if notifications are disabled in settings
        if (!empty($optionInfo['disable_register_product_notification'])) {
            // Emptying the recipient prevents the email from being sent
            $recipient = '';
        }
    }

    return $recipient;
}
add_filter( 'woocommerce_email_recipient_new_order', 'filter_woocommerce_email_recipient_keyzy_new_order', 10, 2 );
add_filter( 'woocommerce_email_recipient_customer_completed_order', 'filter_woocommerce_email_recipient_keyzy_new_order', 10, 2 );

// AJAX Handler registration
add_action('wp_ajax_delete_activation', 'delete_activation_callback');

/**
 * AJAX Callback: Deletes a license activation via the Keyzy API.
 */
function delete_activation_callback() {
    global $connection_array;
    
    // 1. Security Check (Nonce)
    // The third parameter 'false' prevents the script from dying immediately, allowing us to send a JSON response.
    if (!check_ajax_referer('delete_activation_nonce', 'nonce', false)) {
        wp_send_json_error(['message' => 'Invalid nonce']);
        return;
    }
    
    // 2. Authorization Check
    if (!is_user_logged_in()) {
        wp_send_json_error(['message' => 'Unauthorized']);
        return;
    }

    // 3. Retrieve Settings
    $optionInfo = get_option('keyzy_option_name', []);
    $app_id = $optionInfo['app_id'] ?? '';
    $api_key = $optionInfo['api_key'] ?? '';
    
    $activation_id = isset($_POST['activation_id']) ? intval($_POST['activation_id']) : 0;

    if (!$activation_id) {
        wp_send_json_error(['message' => 'Missing activation ID']);
        return;
    }

    try {
        $client = new GuzzleHttp\Client( $connection_array );
        
        // Ensure the backend route is correct: DELETE /activations/{id}
        // Assuming KEYZYURL includes necessary slashes or context.
        $response = $client->request('DELETE', 'activations/' . $activation_id, [
            'form_params' => [
                'app_id' => $app_id,
                'api_key' => $api_key
            ]
        ]);
        
        $body = json_decode($response->getBody());
        $message = isset($body->message) ? $body->message : 'Activation deleted successfully';
        
        wp_send_json_success(['message' => $message]);
        
    } catch (GuzzleHttp\Exception\RequestException $e) {
        // Handle Guzzle HTTP exceptions with response details
        $msg = $e->getMessage();
        if ($e->hasResponse()) {
            $respBody = json_decode($e->getResponse()->getBody());
            if (isset($respBody->message)) {
                $msg = $respBody->message;
            } elseif (isset($respBody->error->message)) {
                $msg = $respBody->error->message;
            }
        }
        wp_send_json_error(['message' => $msg]);
    } catch (Exception $e) {
        // Handle any other exceptions
        wp_send_json_error(['message' => $e->getMessage()]);
    }
}
