<?php
/**
 * Keyzy Downloads - Data Processing Layer
 *
 * This file handles all data fetching and processing without any HTML output.
 * Returns a structured array with all data needed for rendering.
 */

use Automattic\WooCommerce\Utilities\OrderUtil;
use GuzzleHttp\Exception\RequestException;

/**
 * Fetch and process all downloads data for the current user.
 *
 * @return array {
 *     @type bool   $success            Whether data was fetched successfully.
 *     @type string $error              Error message if any.
 *     @type string $error_type         Type of error: 'no_user', 'no_orders', 'invalid_settings', 'api_error', 'no_items'.
 *     @type array  $settings           Plugin settings (show_download_link, show_host_id_input, etc.).
 *     @type array  $items              Array of processed license items ready for rendering.
 *     @type string $shop_url           URL to the shop page.
 * }
 */
function keyzy_get_downloads_data() {
    // Include definitions
    require_once dirname(__FILE__) . '/../defines.php';
    global $connection_array;

    $result = [
        'success'    => false,
        'error'      => '',
        'error_type' => '',
        'settings'   => [],
        'items'      => [],
        'shop_url'   => esc_url( wc_get_page_permalink( 'shop' ) ),
    ];

    // Check if user is logged in
    $current_user_id = get_current_user_id();
    if ( ! $current_user_id ) {
        $result['error_type'] = 'no_user';
        $result['error'] = __( 'Please log in to view your downloads.', 'woocommerce' );
        return $result;
    }

    // Get Keyzy settings
    $optionInfo = get_option( 'keyzy_option_name', [] );
    $app_id  = ( is_array( $optionInfo ) && isset( $optionInfo['app_id'] ) ) ? trim( $optionInfo['app_id'] ) : '';
    $api_key = ( is_array( $optionInfo ) && isset( $optionInfo['api_key'] ) ) ? trim( $optionInfo['api_key'] ) : '';

    // Store settings for rendering
    $result['settings'] = [
        'show_download_link'         => ! empty( $optionInfo['show_download_link'] ),
        'show_host_id_input'         => ! empty( $optionInfo['show_host_id_input'] ),
        'show_activations_and_links' => ! empty( $optionInfo['show_manageable_activations_to_users'] ),
    ];

    // Fetch Orders based on environment (HPOS vs Traditional)
    $customer_orders = null;
    if ( class_exists( 'Automattic\WooCommerce\Utilities\OrderUtil' ) && OrderUtil::custom_orders_table_usage_is_enabled() ) {
        $args = [
            'customer' => $current_user_id,
            'limit'    => -1,
        ];
        $customer_orders = wc_get_orders( $args );
    } else {
        $customer_orders = get_posts( apply_filters( 'woocommerce_my_account_my_orders_query', [
            'numberposts' => -1,
            'meta_key'    => '_customer_user',
            'meta_value'  => $current_user_id,
            'post_type'   => wc_get_order_types( 'view-orders' ),
            'post_status' => array_keys( wc_get_order_statuses() ),
        ] ) );
    }

    // Check if customer has orders
    if ( empty( $customer_orders ) ) {
        $result['error_type'] = 'no_orders';
        return $result;
    }

    // Validate API settings
    if ( strlen( $app_id ) < 8 || strlen( $api_key ) < 8 ) {
        error_log( 'KEYZY Error: Invalid Settings. APP ID and API KEY must be at least 8 characters long. Please check Keyzy Settings.' );
        $result['error_type'] = 'invalid_settings';
        $result['error'] = __( 'Plugin configuration error. Please contact the administrator.', 'woocommerce' );
        return $result;
    }

    // =========================================================================
    // PHASE 1: Collect all items to query in batch
    // =========================================================================
    $queries = [];
    $items_metadata = [];

    foreach ( $customer_orders as $customer_order ) {
        $order = new WC_Order( $customer_order );

        // Only process completed orders
        if ( $order->get_status() !== 'completed' ) {
            continue;
        }

        if ( count( $order->get_items() ) <= 0 ) {
            continue;
        }

        foreach ( $order->get_items() as $keyitem => $item ) {
            $_product_id = $item->get_product_id();
            $_product    = wc_get_product( $_product_id );

            if ( ! $_product ) {
                continue;
            }

            // Create unique item_id for matching response later
            $item_id = 'item_' . $order->get_id() . '_' . $keyitem;

            $queries[] = [
                'item_id'    => $item_id,
                'sku_number' => $_product->get_sku(),
                'user_id'    => $order->get_user_id(),
                'order_id'   => $order->get_id(),
            ];

            // Store metadata for rendering
            $items_metadata[ $item_id ] = [
                'order'    => $order,
                'product'  => $_product,
                'keyitem'  => $keyitem,
            ];
        }
    }

    // If no items to query, return early
    if ( empty( $queries ) ) {
        $result['error_type'] = 'no_items';
        return $result;
    }

    // =========================================================================
    // PHASE 2: Make single batch API call
    // =========================================================================
    $client = new GuzzleHttp\Client( $connection_array );

    // Get WooCommerce version
    $wc_version = defined( 'WC_VERSION' ) ? WC_VERSION : 'unknown';
    $plugin_version = defined( 'KEYZYWC_VERSION' ) ? KEYZYWC_VERSION : '1.0.0';

    $batch_request = [
        'app_id'      => $app_id,
        'api_key'     => $api_key,
        'version'     => '2.0',
        'client_info' => [
            'platform'         => 'woocommerce',
            'platform_version' => $wc_version,
            'plugin_version'   => $plugin_version,
        ],
        'queries' => $queries,
    ];

    $batch_results = [];

    try {
        $response = $client->request( 'POST', 'licenses/query', [
            'json' => $batch_request,
        ] );
        $batch_results = json_decode( $response->getBody(), true );
    } catch ( Exception $e ) {
        error_log( 'KEYZY Error (batch licenses/query): ' . $e->getMessage() );
        $result['error_type'] = 'api_error';
        $result['error'] = __( 'Error fetching license data. Please try again later.', 'woocommerce' );
        return $result;
    }

    // Check for API error
    if ( isset( $batch_results['error'] ) ) {
        error_log( sprintf(
            'KEYZY API Error: %s',
            $batch_results['error']['message'] ?? 'Unknown error'
        ) );
        $result['error_type'] = 'api_error';
        $result['error'] = __( 'Error fetching license data. Please try again later.', 'woocommerce' );
        return $result;
    }

    // =========================================================================
    // PHASE 3: Process results into renderable items
    // =========================================================================

    // Build items array for rendering
    // Iterate API results directly — same item_id may appear multiple times
    // when a customer purchased quantity > 1 of the same SKU.
    $items = [];
    $custom_placeholder_url = plugins_url( 'assets/img/software-placeholder.jpeg', dirname( __FILE__ ) );
    $download_base_url = home_url( '/wp-content/plugins/keyzywc/tpl/dl.php' );
    $license_index = 0;

    $api_results = ( isset( $batch_results['results'] ) && is_array( $batch_results['results'] ) )
        ? $batch_results['results']
        : [];

    foreach ( $api_results as $api_result ) {
        $item_id = $api_result['item_id'] ?? null;

        // Skip if no matching metadata for this item
        if ( ! $item_id || ! isset( $items_metadata[ $item_id ] ) ) {
            continue;
        }

        // Check for item-level error or not found
        if ( isset( $api_result['error'] ) ) {
            error_log( sprintf(
                'KEYZY API Error for item %s: %s',
                $item_id,
                $api_result['error']['message'] ?? 'Unknown'
            ) );
            continue;
        }

        // Skip if status is not_found or license is empty
        if ( ( isset( $api_result['status'] ) && $api_result['status'] === 'not_found' ) || empty( $api_result['license'] ) ) {
            continue;
        }

        $metadata = $items_metadata[ $item_id ];
        $keyitem = $metadata['keyitem'];
        $licenseData = $api_result['license'];

        // Build the item data for rendering
        $item = [
            'item_id'        => $item_id,
            'keyitem'        => $keyitem,
            'image_url'      => ! empty( $licenseData['image_url'] ) ? $licenseData['image_url'] : $custom_placeholder_url,
            'sku_url'        => $licenseData['sku_url'] ?? '',
            'serial'         => $licenseData['serial'] ?? '',
            'products'       => [],
            'activations_error' => $licenseData['activations_error'] ?? null,
        ];

        // Build activation lookup by product_id
        $activations_by_product = [];
        if ( ! empty( $licenseData['activations'] ) ) {
            foreach ( $licenseData['activations'] as $activation ) {
                $pid = $activation['product_id'] ?? null;
                if ( $pid !== null ) {
                    $activations_by_product[ $pid ][] = [
                        'id'         => $activation['id'],
                        'host_id'    => $activation['host_id'],
                        'device_tag' => $activation['device_tag'],
                        'nonce'      => wp_create_nonce( 'delete_activation_nonce' ),
                    ];
                }
            }
        }

        // Process products with their activations grouped together
        if ( ! empty( $licenseData['products'] ) ) {
            foreach ( $licenseData['products'] as $itemCnt => $product ) {
                $unique_dom_id = esc_attr( $product['code'] . "_id_" . $keyitem . "_" . $license_index . "_" . $itemCnt );
                $pid = $product['product_id'] ?? null;

                $item['products'][] = [
                    'code'          => $product['code'],
                    'name'          => $product['name'],
                    'filename'      => $product['filename'],
                    'unique_dom_id' => $unique_dom_id,
                    'download_url'  => add_query_arg( [
                        's'       => $licenseData['serial'],
                        'c'       => $product['code'],
                        'fn'      => $product['filename'],
                        'host_id' => ''
                    ], $download_base_url ),
                    'base_url'      => $download_base_url,
                    'activations'   => $activations_by_product[ $pid ] ?? [],
                ];
            }
        }

        $items[] = $item;
        $license_index++;
    }

    // Check if we have any valid items
    if ( empty( $items ) ) {
        $result['error_type'] = 'no_items';
        return $result;
    }

    // Success!
    $result['success'] = true;
    $result['items'] = $items;

    return $result;
}