<?php
/**
 * Keyzy Downloads - Template/View Layer
 *
 * This file handles HTML rendering only.
 * All data processing is done in keyzy-downloads-data.php
 */

// Include the data processing layer
require_once dirname(__FILE__) . '/keyzy-downloads-data.php';

// =========================================================================
// PHASE 1: Fetch all data (no output happens here)
// =========================================================================
$downloads_data = keyzy_get_downloads_data();

// =========================================================================
// PHASE 2: Enqueue scripts
// =========================================================================
if ( ! wp_script_is( 'axios', 'enqueued' ) ) {
    wp_enqueue_script( 'axios', 'https://unpkg.com/axios/dist/axios.min.js', [], '1.0', true );
}

// =========================================================================
// PHASE 3: Render HTML (all data is ready, safe to output)
// =========================================================================
?>
<div id="downloads-wrapper">
<?php

// Handle different states based on data result
if ( ! $downloads_data['success'] ) :

    switch ( $downloads_data['error_type'] ) {
        case 'no_user':
            echo '<p>' . esc_html( $downloads_data['error'] ) . '</p>';
            break;

        case 'no_orders':
        case 'no_items':
            ?>
            <p><?php printf(
                __( "You do not have any products available at the moment. Please visit our <a href='%s'>Store</a> to browse the selection of products.", 'woocommerce' ),
                esc_url( $downloads_data['shop_url'] )
            ); ?></p>
            <?php
            break;

        case 'invalid_settings':
        case 'api_error':
            echo '<p>' . esc_html( $downloads_data['error'] ) . '</p>';
            break;

        default:
            echo '<p>' . esc_html__( 'Unable to load downloads. Please try again later.', 'woocommerce' ) . '</p>';
            break;
    }

else :
    // Success - render the downloads
    $settings = $downloads_data['settings'];
    $items = $downloads_data['items'];
    ?>

    <style>
        .keyzy-sku-block {
            border-top: 2px solid #ccc;
            margin-top: 20px;
            padding-top: 15px;
        }
        .keyzy-sku-block:first-child {
            border-top: none;
            margin-top: 0;
        }
        .keyzy-product-block {
            border-top: 1px solid #eee;
            margin-top: 12px;
            padding-top: 10px;
        }
        .keyzy-activation-row {
            margin-top: 8px;
            padding-top: 5px;
            margin-left: 10px;
        }
    </style>

    <h2><?php echo esc_html__( 'My Products', 'woocommerce' ); ?></h2><br>
    <div class="digital-downloads">

    <?php foreach ( $items as $item ) : ?>
        <div class="keyzy-sku-block">
        <table>
            <tr>
                <td colspan="2"></td>
                <td>
                    <img src="<?php echo esc_url( $item['image_url'] ); ?>"
                         width="320" height="240"
                         class="keyzy-product-img"
                         alt="<?php echo esc_attr__( 'Product License Image', 'woocommerce' ); ?>"/><br>
                </td>
                <td>
                    <?php if ( ! empty( $item['sku_url'] ) ) : ?>
                        <a href="<?php echo esc_url( $item['sku_url'] ); ?>" target="_blank">
                            <?php echo esc_html__( 'To Download Go To Product Page', 'woocommerce' ); ?>
                        </a><br>
                    <?php endif; ?>

                    <b><?php echo esc_html( 'Product Serial:' ); ?></b> <?php echo esc_html( $item['serial'] ); ?><br>

                    <?php if ( ! empty( $item['products'] ) && ( $settings['show_download_link'] || $settings['show_activations_and_links'] ) ) : ?>
                        <?php foreach ( $item['products'] as $product ) : ?>
                            <div class="keyzy-product-block">
                            <b><?php echo esc_html( $product['name'] ); ?></b>

                            <?php if ( $settings['show_download_link'] ) : ?>
                                <?php if ( $settings['show_host_id_input'] ) : ?>
                                    <div class="keyzy-download-block">
                                        <br><?php esc_html_e( 'Host ID:', 'woocommerce' ); ?>
                                        <input type="text"
                                               id="host_id_<?php echo esc_attr( $product['unique_dom_id'] ); ?>"
                                               class="keyzy-host-id-input"
                                               maxlength="128"
                                               size="36"
                                               data-target-link="link_<?php echo esc_attr( $product['unique_dom_id'] ); ?>"
                                        />
                                    </div>
                                <?php endif; ?>

                                <br>
                                <a href="<?php echo esc_url( $product['download_url'] ); ?>"
                                   id="link_<?php echo esc_attr( $product['unique_dom_id'] ); ?>"
                                   class="keyzy-download-link"
                                   data-base-url="<?php echo esc_url( $product['base_url'] ); ?>"
                                   data-serial="<?php echo esc_attr( $item['serial'] ); ?>"
                                   data-code="<?php echo esc_attr( $product['code'] ); ?>"
                                   data-filename="<?php echo esc_attr( $product['filename'] ); ?>"
                                   data-version="<?php echo defined('KEYZYWC_VERSION') ? esc_attr(KEYZYWC_VERSION) : ''; ?>"
                                   data-host="<?php echo esc_attr( parse_url( home_url(), PHP_URL_HOST ) ); ?>"
                                >
                                    <?php printf( esc_html__( 'Download %s License File', 'woocommerce' ), esc_html( $product['name'] ) ); ?>
                                </a>
                            <?php endif; ?>

                            <?php if ( $settings['show_activations_and_links'] && ! empty( $product['activations'] ) ) : ?>
                                <?php foreach ( $product['activations'] as $activation ) : ?>
                                    <div class="keyzy-activation-row">
                                        <table>
                                            <tr>
                                                <td><b><?php esc_html_e( 'Activation', 'woocommerce' ); ?></b></td>
                                                <td><?php esc_html_e( 'Host ID', 'woocommerce' ); ?></td>
                                                <td><?php echo esc_html( $activation['host_id'] ); ?></td>
                                            </tr>
                                            <tr>
                                                <td><b><?php esc_html_e( 'Device Tag', 'woocommerce' ); ?></b></td>
                                                <td><?php echo esc_html( $activation['device_tag'] ); ?></td>
                                            </tr>
                                            <tr>
                                                <td colspan="2">
                                                    <a href="#"
                                                       class="keyzy-delete-activation"
                                                       data-activation-id="<?php echo esc_attr( $activation['id'] ); ?>"
                                                       data-host-id="<?php echo esc_attr( $activation['host_id'] ); ?>"
                                                       data-serial="<?php echo esc_attr( $item['serial'] ); ?>"
                                                       data-nonce="<?php echo esc_attr( $activation['nonce'] ); ?>"
                                                    >
                                                        <?php esc_html_e( 'Delete Activation', 'woocommerce' ); ?>
                                                    </a>
                                                </td>
                                            </tr>
                                        </table>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>

                    <?php if ( isset( $item['activations_error'] ) ) : ?>
                        <?php if ( $item['activations_error'] === 'Activation does not exist!' ) : ?>
                            <br><br><b><?php echo esc_html__( 'There is no activation for this license', 'woocommerce' ); ?></b>
                        <?php else : ?>
                            <br><br><b><?php echo esc_html__( 'Please report that!', 'woocommerce' ); ?></b>
                            <?php echo esc_html( 'Error: ' . $item['activations_error'] ); ?>
                        <?php endif; ?>
                    <?php endif; ?>

                </td>
            </tr>
        </table>
        </div>
    <?php endforeach; ?>

    </div>

<?php endif; ?>
</div>

<script type="text/javascript">
document.addEventListener("DOMContentLoaded", function() {

    // --- Logic for Host ID Input and Download Link ---

    // PHP variables to JS
    const hostIdRegexPattern = '<?php echo defined('HOST_ID_REGEX') ? addslashes(HOST_ID_REGEX) : ''; ?>';
    const hostIdLength = <?php echo defined('HOST_ID_LENGTH') ? (int)HOST_ID_LENGTH : 12; ?>;
    const hostIdRegex = hostIdRegexPattern ? new RegExp(hostIdRegexPattern) : null;

    const hostInputs = document.querySelectorAll('.keyzy-host-id-input');

    hostInputs.forEach(function(input) {
        input.addEventListener('keyup', function() {
            const val = this.value.trim();
            const targetLinkId = this.getAttribute('data-target-link');
            const linkElement = document.getElementById(targetLinkId);

            if (!linkElement) return;

            // Retrieve data attributes
            const baseUrl = linkElement.getAttribute('data-base-url');
            const serial = linkElement.getAttribute('data-serial');
            const code = linkElement.getAttribute('data-code');
            const fn = linkElement.getAttribute('data-filename');
            const version = linkElement.getAttribute('data-version');
            const host = linkElement.getAttribute('data-host');

            // Construct device tag
            const deviceTag = "KEYZYWc v:" + version + " " + host;

            // Construct URL
            const finalUrl = baseUrl +
                "?s=" + encodeURIComponent(serial) +
                "&c=" + encodeURIComponent(code) +
                "&fn=" + encodeURIComponent(fn) +
                "&host_id=" + encodeURIComponent(val) +
                "&device_tag=" + encodeURIComponent(deviceTag);

            // Validation logic
            let isValid = false;
            if (hostIdRegex) {
                isValid = hostIdRegex.test(val);
            } else {
                isValid = val.length > 0;
            }

            if (isValid) {
                linkElement.href = finalUrl;
                this.style.borderColor = ''; // Valid
                linkElement.style.pointerEvents = 'auto';
                linkElement.classList.remove('disabled-link');
            } else {
                linkElement.href = '#';
                this.style.borderColor = 'red'; // Invalid
                linkElement.classList.add('disabled-link');
            }
        });
    });

    // Intercept clicks on download links if validation fails
    document.querySelectorAll('.keyzy-download-link').forEach(function(link) {
        link.addEventListener('click', function(e) {
            if (this.getAttribute('href') === '#') {
                e.preventDefault();
                alert("Please check the Host ID field format.");
            }
        });
    });

    // --- Logic for Delete Activation ---

    const deleteButtons = document.querySelectorAll('.keyzy-delete-activation');

    deleteButtons.forEach(function(btn) {
        btn.addEventListener('click', function(e) {
            e.preventDefault();

            const activationId = this.getAttribute('data-activation-id');
            const hostId = this.getAttribute('data-host-id');
            const serial = this.getAttribute('data-serial');
            const nonce = this.getAttribute('data-nonce');

            const confirmMsg = "Are you sure you want to delete the activation for Host ID: " + hostId + " for Product Serial: " + serial + "?";

            if (!confirm(confirmMsg)) {
                return;
            }

            // UI Update
            const originalText = this.textContent;
            this.textContent = 'Deleting...';
            this.style.pointerEvents = 'none';
            this.style.opacity = '0.5';

            const params = new URLSearchParams();
            params.append('action', 'delete_activation');
            params.append('activation_id', activationId);
            params.append('nonce', nonce);

            // Using the global axios object (assuming it's loaded)
            if (typeof axios !== 'undefined') {
                axios.post('<?php echo admin_url( 'admin-ajax.php' ); ?>', params)
                    .then(response => {
                        console.log(response.data);
                        if(response.data && response.data.data) {
                            alert(response.data.data.message || 'Success');
                        } else {
                            alert('Operation successful.');
                        }
                        location.reload();
                    })
                    .catch(error => {
                        console.error(error);
                        let msg = 'Unknown Error';
                        if (error.response && error.response.data && error.response.data.data) {
                            msg = error.response.data.data.message;
                        }
                        alert('Error: ' + msg);

                        // Reset UI
                        this.textContent = originalText;
                        this.style.pointerEvents = 'auto';
                        this.style.opacity = '1';
                    });
            } else {
                alert("Axios library is not loaded.");
            }
        });
    });
});
</script>